<?php
require 'server/config.db.php';
session_start();

if (!isset($_SESSION['username'])) {
    header('Location: login.php');
    exit();
}

// Get the user_id from the query string
$user_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : 0;

if ($user_id === 0) {
    echo "Invalid user ID.";
    exit();
}

// Fetch all messages for the specific user
try {
    $query = "SELECT sender, sender_name, content, created_at FROM messages WHERE user_id = :user_id ORDER BY created_at ASC";
    $stmt = $pdo->prepare($query);
    $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $stmt->execute();
    $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Fetch the sender_name from the latest message
    $sender_name = $messages ? $messages[0]['sender_name'] : 'User';
} catch (PDOException $e) {
    echo "Query failed: " . $e->getMessage();
    exit();
}

// Handle form submission for sending a new message
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $reply_content = trim($_POST['reply_content']);

    if (!empty($reply_content)) {
        try {
            $insertQuery = "INSERT INTO messages (user_id, sender, sender_name, content, created_at) VALUES (:user_id, 'admin', :sender_name, :content, NOW())";
            $insertStmt = $pdo->prepare($insertQuery);
            $insertStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $insertStmt->bindParam(':sender_name', $sender_name, PDO::PARAM_STR);
            $insertStmt->bindParam(':content', $reply_content, PDO::PARAM_STR);
            $insertStmt->execute();

            // Redirect to the same page to avoid form resubmission issues
            header("Location: view_message.php?user_id=$user_id");
            exit();
        } catch (PDOException $e) {
            echo "Failed to send message: " . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Messages</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" rel="stylesheet">
    <link href="style.css" rel="stylesheet">
    <style>
        .message {
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 10px;
        }
        .message.admin {
            background-color: #e3f2fd; /* Light blue background for admin messages */
            text-align: right; /* Align admin messages to the right */
        }
        .message.user {
            background-color: #f1f1f1; /* Light gray background for user messages */
            text-align: left; /* Align user messages to the left */
        }
    </style>
</head>
<body>
    <?php require 'server/active-check.php' ?>
    <?php require 'server/nav.php'; ?>
    
    <!-- Sidebar trigger button for mobile -->
    <button class="btn btn-primary sidebar-toggle-btn d-md-none" type="button" data-bs-toggle="offcanvas" data-bs-target="#offcanvasSidebar" aria-controls="offcanvasSidebar">
        <i class="fas fa-bars"></i>
    </button>

    <!-- Sidebar -->
    <?php require 'server/sidebar.php'; ?>

    <div class="container-fluid mt-5">
        <div class="row">
            <div class="col-12">
                <h2 class="mb-4">Messages with User ID: <?php echo htmlspecialchars($user_id); ?></h2>

                <div class="card">
                    <div class="card-body">
                        <div class="messages-container">
                            <?php if (!empty($messages)): ?>
                                <?php foreach ($messages as $message): ?>
                                    <div class="message <?php echo $message['sender'] === 'admin' ? 'admin' : 'user'; ?>">
                                        <h5><?php echo htmlspecialchars($message['sender_name']); ?></h5>
                                        <p><?php echo nl2br(htmlspecialchars($message['content'])); ?></p>
                                        <p class="text-muted"><small>Sent on: <?php echo htmlspecialchars($message['created_at']); ?></small></p>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <p>No messages found.</p>
                            <?php endif; ?>
                        </div>

                        <form method="POST" class="mt-4">
                            <div class="mb-3">
                                <label for="reply_content" class="form-label">Reply:</label>
                                <textarea class="form-control" id="reply_content" name="reply_content" rows="4" required></textarea>
                            </div>
                            <button type="submit" class="btn btn-primary">Send Reply</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
</body>
</html>
